<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Resource;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Storage;


class ResourceController extends Controller
{
    public function registerresource(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name_resource' => 'required|max:100',
            'description_resource' => 'required|max:300',
            'resource' => 'required|file|mimes:pdf,doc,docx,xls,xlsx,ppt,pptx|max:10240',
            'id_class' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Error en la validación',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        try {
            $uploadedFile = $request->file('resource');

            $resourceName = time().'_'.$uploadedFile->getClientOriginalName();

            $resourcePath = $uploadedFile->storeAs('public/resources', $resourceName);

            if (!$resourcePath) {
                throw new \Exception("Error al guardar el archivo");
            }

            $relativePath = str_replace('public/', '', $resourcePath);

            $resource = Resource::create([
                'name_resource' => $request->name_resource,
                'description_resource' => $request->description_resource,
                'resource' => $relativePath,
                'id_class' => $request->id_class
            ]);

            return response()->json([
                'message' => 'Recurso registrado correctamente',
                'data' => $resource,
                'file_url' => asset("storage/".$relativePath)
            ], Response::HTTP_OK);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error en el servidor',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    public function showresource($id_resource)
    {

        $id_resource = Resource::find($id_resource);
        if (!$id_resource) {

            return response()->json(
                ['message' => 'No se encontro el recurso'],
                Response::HTTP_NOT_FOUND
            );
        }

        return response()->json(
            ['message' => $id_resource],
            Response::HTTP_OK
        );
    }

    public function listresource(Request $request)
    {

        $perPage = $request->input('per_page', 20);

        $resource = Resource::paginate($perPage);

        if ($resource->isEmpty()) {
            return response()->json(['message' => 'No hay recursos registrados'], Response::HTTP_NOT_FOUND);
        }

        return response()->json($resource, Response::HTTP_OK);
    }

    public function upgradepartialresource(Request $request, $id_resource)
    {
        $validator = Validator::make($request->all(), [
            'name_resource' => 'sometimes|required|max:100',
            'description_resource' => 'sometimes|required|max:300',
            'resource' => 'sometimes|file|mimes:pdf,doc,docx,xls,xlsx,ppt,pptx|max:10240',
            'id_class' => 'sometimes|required|exists:class,id_class'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Error en la validación',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        try {
            $resource = Resource::findOrFail($id_resource);

            $resource->fill($request->only(['name_resource', 'description_resource', 'id_class']));

            if ($request->hasFile('resource')) {
               
                if ($resource->resource && Storage::disk('public')->exists($resource->resource)) {
                    Storage::disk('public')->delete($resource->resource);
                }

                $uploadedFile = $request->file('resource');
                $resourceName = time().'_'.$uploadedFile->getClientOriginalName();
                $resourcePath = $uploadedFile->storeAs('public/resources', $resourceName);
                $relativePath = str_replace('public/', '', $resourcePath);

                $resource->resource = $relativePath;
            }

            $resource->save();

            return response()->json([
                'message' => 'Recurso actualizado correctamente',
                'data' => $resource,
                'file_url' => $resource->resource ? asset("storage/".$resource->resource) : null
            ], Response::HTTP_OK);

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'message' => 'Recurso no encontrado'
            ], Response::HTTP_NOT_FOUND);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error en el servidor',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    // En tu controlador
    public function getProductById($id)
    {
        try {
            $product = Product::where('product_id', $id)->first();

            if (!$product) {
                return response()->json([
                    'success' => false,
                    'message' => 'Producto no encontrado'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $product->makeVisible($product->getVisible())
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al buscar el producto',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}


