<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ClassLife;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Storage;

class ClassLifeController extends Controller
{
    public function registerclass(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'title' => 'required',
                'description' => 'required',
                'image_class' => 'required|string',
                'url_video' => 'required|string',
                'id_module' => 'required'
            ]
        );

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar la clase',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Procesar la imagen
        $imagePath = null;
        $imageName = null;

        if ($request->has('image_class') && !empty($request->image_class)) {
            $imageBase64 = $request->image_class;

            // Verificar si la imagen es Base64 válida
            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                // Extraer el tipo de imagen (png, jpeg, etc.)
                $imageType = $type[1];

                // Eliminar el prefijo "data:image/<tipo>;base64,"
                $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);

                // Decodificar la imagen Base64
                $imageData = base64_decode($imageBase64);

                // Verificar si la decodificación fue exitosa
                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                // Generar un nombre único para la imagen
                $imageName = uniqid() . '.' . $imageType;

                // Guardar la imagen en el storage
                $imagePath = 'images/' . $imageName;
                Storage::disk('public')->put($imagePath, $imageData);
            } else {
                return response()->json(
                    ['message' => 'El formato de la imagen no es válido'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }
        $class = ClassLife::create([
            'title' => $request->title,
            'description' => $request->description,
            'image_class' => $imagePath, // Guardar la ruta de la imagen
            'url_video' => $request->url_video,
            'id_module' => $request->id_module
        ]);

        if (!$class) {
            return response()->json(
                ['message' => 'Error al guardar el dato'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        return response()->json(
            [
                'message' => 'la clase se registró correctamente',
                'data' => $class
            ],
            Response::HTTP_CREATED
        );
    }

    public function showclass($id_class)
    {
    $class = ClassLife::find($id_class);

    if (!$class) {
        return response()->json([
            'message' => 'Clase no encontrada',
        ], Response::HTTP_NOT_FOUND);
    }

    // Obtener la ruta de la imagen
    $imagePath = $class->image_class;

    // Verificar si la imagen existe en el almacenamiento
    if ($imagePath && Storage::disk('public')->exists($imagePath)) {
        // Leer la imagen del almacenamiento
        $imageData = Storage::disk('public')->get($imagePath);

        // Convertir la imagen a Base64
        $imageBase64 = base64_encode($imageData);

        // Agregar la imagen en Base64 a la respuesta
        $class->image_base64 = $imageBase64;
    } else {
        // Si no hay imagen, devolver null
        $class->image_base64 = null;
    }

    // Devolver la respuesta con el curso y la imagen en Base64
    return response()->json([
        'message' => 'Clase encontrada',
        'data' => $class,
    ], Response::HTTP_OK);

    }

    public function listclass(Request $request)
    {
        $perPage = $request->input('per_page', 20);

        $class = ClassLife::paginate($perPage);

        if ($class->isEmpty()) {
            return response()->json(['message' => 'No hay clases registradas'], Response::HTTP_NOT_FOUND);
        }

        return response()->json($class, Response::HTTP_OK);
    }

    public function upgradepartialclass(Request $request, $id_class)
    {

        //busqueda por id
        $class = ClassLife::find($id_class);
        if (!$class) {

            return response()->json(
                ['message' => 'No se encontro la clase'],
                Response::HTTP_NOT_FOUND
            );
        }

        $validator = Validator::make($request->all(), [
           'title' => 'sometimes|max:100',
            'description' => 'sometimes|max:200',
            'url_video' => 'nullable',
            'id_module' => 'nullable'
        ]);

        //si los datos son invalidos mensaje de error
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar la clase',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }
        $updatedFields = 0;

        // Manejo de la imagen (solo si se envía y no está vacía)
        if ($request->filled('image_class')) {
            $imageBase64 = $request->image_class;

            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                $imageData = base64_decode(substr($imageBase64, strpos($imageBase64, ',') + 1));

                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                $imageName = uniqid().'.'.$type[1];
                $imagePath = 'images/courses/'.$imageName;

                // Eliminar imagen anterior si existe
                if ($class->image_class && Storage::disk('public')->exists($class->image_class)) {
                    Storage::disk('public')->delete($class->image_class);
                }

                // Guardar nueva imagen
                Storage::disk('public')->put($imagePath, $imageData);

                // Actualizar solo campos existentes
                $class->image_class = $imagePath;
                $class->name_img_class = $imageName;

                $updatedFields++;
            } else {
                return response()->json(
                    ['message' => 'La imagen no está en formato Base64 válido'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        $errors = [];
        $fields = [
          'title',
          'description',
          'url_video',
          'id_module'
      ];

      foreach ($fields as $field) {
        if ($request->has($field) && empty($request->$field)) {
            $errors[$field] = 'El campo ' . $field . ' no puede estar vacío.';
        }
    }

    if (!empty($errors)) {
        return response()->json(
            [
                'message' => 'Error al actualizar el evento: campos vacíos',
                'errors' => $errors,
            ],
            Response::HTTP_BAD_REQUEST
        );
    }

        $dataToUpdate = array_filter($request->only([
          'title',
          'description',
          'url_video',
          'id_module'
         ]), function ($value) {
             return !empty($value);
         });


        $class->fill($dataToUpdate);
        $class->save();

        return response()->json(
            ['message' => $class],
            Response::HTTP_OK
        );

    }

    //revisar si se esta usando la ruta correcta
    public function getClassById($id_class)
    {
        $class = ClassLife::with('video')->find($id_class);

        if (!$class) {
            return response()->json([
                'error' => 'Class not found'
            ], 404);
        }

        return response()->json([
            'message' => [
                'id_class' => $class->id_class,
                'title' => $class->title,
                'description' => $class->description,
                'id_user' => $class->id_user,
                'url_video' => $class->url_video,
                'video_info' => [
                    'nombre_video' => $class->video->nombre_video,
                    'description_video' => $class->video->description_video,
                    'url_video' => $class->video->url_video,
                    'status_video' => $class->video->status_video,
                ],
                'id_module' => $class->id_module,
                'created_at' => $class->created_at,
                'updated_at' => $class->updated_at,
            ]
        ]);
    }
}














