<?php

namespace App\Http\Controllers\Api;

use Illuminate\Support\Facades\Cache;
use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Storage;
use App\Models\ExternalUser;
use Exception;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;



class UserController extends Controller
{
    public function listUser(Request $request)
    {
        $users = User::paginate($request->input('per_page', 10));

        if ($users->isEmpty()) {
            return response()->json([
                'message' => 'No se encontraron usuarios',
            ], Response::HTTP_NOT_FOUND);
        }

        $users->getCollection()->transform(function ($user) {
            return $this->processUserImage($user);
        });

        return response()->json([
            'message' => 'Usuarios encontrados',
            'data' => $users,
        ], Response::HTTP_OK);
    }

    /**
     * Registro público (sin autenticación requerida)
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nombre_user' => 'required|max:100',
            'apellido_user' => 'required|max:100',
            'ci_user' => 'required|max:30|unique:users_life',
            'correo_user' => 'required|max:100|email|unique:users_life',
            'password_user' => 'required|max:61|min:8',
            'image_user' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Error en los datos del registro',
                'errors' => $validator->errors()
            ], Response::HTTP_BAD_REQUEST);
        }

        // Procesar imagen si está presente
        $imageData = $this->processImage($request->image_user);

        $user = User::create([
            'nombre_user' => $request->nombre_user,
            'apellido_user' => $request->apellido_user,
            'ci_user' => $request->ci_user,
            'correo_user' => $request->correo_user,
            'password_user' => Hash::make($request->password_user),
            'name_img_user' => $imageData['image_name'] ?? null,
            'image_user' => $imageData['image_path'] ?? null,
            'base64_image' => $request->image_user ?? null,

        ]);

        if (!$user) {
            return response()->json(
                ['message' => 'Error al crear el usuario'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        return response()->json([
            'message' => 'Usuario registrado exitosamente',
            'data' => $this->processUserImage($user)
        ], Response::HTTP_CREATED);
    }

    /**
     * Registro protegido (requiere autenticación Sanctum)
     */
    public function registerUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nombre_user' => 'required|max:100',
            'apellido_user' => 'required|max:100',
            'ci_user' => 'required|max:30|unique:users_life',
            'correo_user' => 'required|max:100|email|unique:users_life',
            'password_user' => 'required|max:61|min:8',
            'status_user' => 'required|max:50',
            'image_user' => 'required|string',
            'id_paises' => 'required|exists:paises_life,id_paises',
            'id_permissions' => 'required|exists:permissions_life,id_permissions',
            'id_courses' => 'required|exists:courses_life,id_courses',
            'id_students' => 'required|exists:students_life,id_students'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Error al registrar el usuario',
                'errors' => $validator->errors()
            ], Response::HTTP_BAD_REQUEST);
        }

        // Procesar imagen
        $imageData = $this->processImage($request->image_user);
        if (!$imageData && $request->image_user) {
            return response()->json([
                'message' => 'El formato de la imagen no es válido'
            ], Response::HTTP_BAD_REQUEST);
        }

        $user = User::create([
            'nombre_user' => $request->nombre_user,
            'apellido_user' => $request->apellido_user,
            'ci_user' => $request->ci_user,
            'correo_user' => $request->correo_user,
            'password_user' => Hash::make($request->password_user),
            'status_user' => $request->status_user,
            'name_img_user' => $imageData['image_name'],
            'image_user' => $imageData['image_path'],
            'base64_image' => $request->image_user,
            'id_paises' => $request->id_paises,
            'id_permissions' => $request->id_permissions,
            'id_courses' => $request->id_courses,
            'id_students' => $request->id_students
        ]);

        if (!$user) {
            return response()->json(
                ['message' => 'Error al guardar el usuario'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        return response()->json([
            'message' => 'Usuario registrado correctamente',
            'data' => $this->processUserImage($user)
        ], Response::HTTP_CREATED);
    }

    public function showUser($id_user)
    {
        $user = User::find($id_user);

        if (!$user) {
            return response()->json([
                'message' => 'Usuario no encontrado',
            ], Response::HTTP_NOT_FOUND);
        }

        return response()->json([
            'message' => 'Usuario encontrado',
            'data' => $this->processUserImage($user),
        ], Response::HTTP_OK);
    }

    public function upgradePartialUser(Request $request, $id_user)
    {
        $user = User::find($id_user);
        if (!$user) {
            return response()->json(
                ['message' => 'No se encontró el usuario'],
                Response::HTTP_NOT_FOUND
            );
        }

        $validator = Validator::make($request->all(), [
            'nombre_user' => 'sometimes|max:100',
            'apellido_user' => 'sometimes|max:100',
            'ci_user' => 'sometimes|max:30|unique:users_life,ci_user,' . $id_user,
            'correo_user' => 'sometimes|max:100|email|unique:users_life,correo_user,' . $id_user,
            'password_user' => 'sometimes|max:61|min:8',
            'status_user' => 'sometimes|max:50',
            'image_user' => 'nullable|string',
            'id_paises' => 'sometimes|exists:paises,id',
            'id_permissions' => 'sometimes|exists:permissions,id',
            'id_courses' => 'sometimes|exists:courses,id',
            'id_students' => 'sometimes|exists:students,id'
        ]);

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el usuario',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Manejo de imagen
        if ($request->has('image_user') && $request->image_user) {
            $imageData = $this->processImage($request->image_user);
            if (!$imageData) {
                return response()->json(
                    ['message' => 'El formato de la imagen no es válido'],
                    Response::HTTP_BAD_REQUEST
                );
            }

            // Eliminar imagen anterior si existe
            if ($user->image_user && Storage::disk('public')->exists($user->image_user)) {
                Storage::disk('public')->delete($user->image_user);
            }

            $user->name_img_user = $imageData['image_name'];
            $user->image_user = $imageData['image_path'];
            $user->base64_image = $request->image_user;
        }

        // Actualizar campos
        $user->fill($request->only([
            'nombre_user',
            'apellido_user',
            'ci_user',
            'correo_user',
            'status_user',
            'id_paises',
            'id_permissions',
            'id_courses',
            'id_students'
        ]));

        // Actualizar contraseña si se proporciona (CORRECCIÓN APLICADA AQUÍ)
        if ($request->has('password_user')) {
            $user->password_user = Hash::make($request->password_user);
        }

        if (!$user->save()) {
            return response()->json(
                ['message' => 'Error al actualizar el usuario'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        return response()->json([
            'message' => 'Usuario actualizado correctamente',
            'data' => $this->processUserImage($user)
        ], Response::HTTP_OK);
    }
    public function validateLoginUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'correo_user' => 'required|email',
            'password_user' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validación fallida',
                'errors' => $validator->errors()
            ], Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $user = User::where('correo_user', $request->correo_user)->first();

        if (!$user || !Hash::check($request->password_user, $user->password_user)) {
            return response()->json([
                'success' => false,
                'message' => 'Credenciales incorrectas'
            ], Response::HTTP_UNAUTHORIZED);
        }

        $token = $user->createToken(
            'login_user_token_' . $user->nombre_user,
            [],
            now()->addMinutes(config('sanctum.expiration'))
        )->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Bienvenido ' . $user->nombre_user,
            'data' => $this->processUserImage($user),
            'access_token' => $token,
            'token_type' => 'Bearer',
            'expires_in' => config('sanctum.expiration') * 60,
        ], Response::HTTP_OK);
    }

    public function logoutUser(Request $request)
    {
        if (!Auth::check()) {
            return response()->json(
                ['message' => 'No estás autenticado.'],
                Response::HTTP_UNAUTHORIZED
            );
        }

        $request->user()->currentAccessToken()->delete();

        return response()->json(
            ['message' => 'Sesión cerrada correctamente'],
            Response::HTTP_OK
        );
    }

    public function showUserCi($ci_user)
    {
        $user = User::where('ci_user', $ci_user)->first();

        if (!$user) {
            return response()->json(
                ['message' => 'Usuario no encontrado'],
                Response::HTTP_NOT_FOUND
            );
        }

        return response()->json([
            'message' => 'Usuario encontrado',
            'data' => $this->processUserImage($user)
        ], Response::HTTP_OK);
    }

    public function showCorreouser($correo_user)
    {
        $user = User::where('correo_user', $correo_user)->first();

        if (!$user) {
            return response()->json(
                ['message' => 'Usuario no encontrado'],
                Response::HTTP_NOT_FOUND
            );
        }

        return response()->json([
            'message' => 'Usuario encontrado',
            'data' => $this->processUserImage($user)
        ], Response::HTTP_OK);
    }

    public function getUserCourses($id_user)
    {
        $user = User::with('courses')->findOrFail($id_user);
        return response()->json([
            'user' => $this->processUserImage($user),
            'courses' => $user->courses
        ]);
    }

    /**
     * Métodos auxiliares
     */
    private function processImage($imageBase64)
    {
        if (!$imageBase64) {
            return null;
        }

        if (!preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
            return null;
        }

        $imageType = $type[1];
        $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);
        $imageData = base64_decode($imageBase64);

        if ($imageData === false) {
            return null;
        }

        $imageName = uniqid() . '.' . $imageType;
        $imagePath = 'images/users/' . $imageName;
        Storage::disk('public')->put($imagePath, $imageData);

        return [
            'image_name' => $imageName,
            'image_path' => $imagePath
        ];
    }

    private function processUserImage($user)
    {
        if (!$user) {
            return $user;
        }

        // URL completa de la imagen
        $user->image_url = $user->image_user ? url(Storage::url($user->image_user)) : null;

        // Generar Base64 si no está guardado pero existe la imagen
        if (empty($user->base64_image) && $user->image_user && Storage::disk('public')->exists($user->image_user)) {
            $imageData = Storage::disk('public')->get($user->image_user);
            $user->base64_image = 'data:image/' . pathinfo($user->image_user, PATHINFO_EXTENSION) . ';base64,' . base64_encode($imageData);
        }

        return $user;
    }

    public function countUserExternal(Request $request)
    {
        // Filtra solo usuarios externos con plan_id = 1
        $count = ExternalUser::where('plan_id', '>', 0)->count();

        return response()->json([
            'message' => 'Conteo de usuarios externos con plan_id = 1',
            'total_users' => $count,
        ], Response::HTTP_OK);
    }

    public function countActiveUsers()
    {
        // Duración del caché (en segundos). Ej: 5 minutos (300 segundos).
        $cacheDuration = 300;

        // Obtener el conteo desde el caché o ejecutar la consulta
        $activeUsersCount = Cache::remember('active_users_count', $cacheDuration, function () {
            // Tokens válidos: creados o usados en las últimas 24 horas
            $threshold = now()->subHours(24);

            return DB::table('personal_access_tokens')
                ->where(function ($query) use ($threshold) {
                    $query->where('last_used_at', '>=', $threshold) // Usados recientemente
                        ->orWhere('created_at', '>=', $threshold); // Tokens nuevos no usados aún
                })
                ->distinct('tokenable_id') // Usuarios únicos
                ->count('tokenable_id');
        });

        return response()->json([
            'active_users_count' => $activeUsersCount,
            'last_updated' => now()->toDateTimeString(),
        ]);
    }

    public function monthlyStats(Request $request)
    {
        $query = DB::table('personal_access_tokens');

        // Filtros por fecha
        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        $stats = $query
            ->select(
                DB::raw('YEAR(created_at) as year'),
                DB::raw('MONTH(created_at) as month'),
                DB::raw('MONTHNAME(created_at) as month_name'),
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('year', 'month', 'month_name')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->get();

        // Calcular total
        $total = $stats->sum('count');

        return response()->json([
            'success' => true,
            'data' => $stats,
            'total' => $total,
            'chart_data' => [
                'labels' => $stats->map(fn($item) => "{$item->month_name} {$item->year}"),
                'datasets' => [
                    [
                        'label' => 'Tokens creados',
                        'data' => $stats->pluck('count')
                    ]
                ]
            ]
        ]);
    }


    public function enlistuser()
    {
        try {
            // Obtener solo los campos username y email de todos los usuarios
            $users = User::query()
                ->select(['username', 'email'])
                ->get();

            return response()->json([
                'success' => true,
                'data' => $users,
                'message' => 'Lista de usuarios obtenida correctamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener los usuarios: ' . $e->getMessage()
            ], 500);
        }
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user' => 'required|string',
            'password' => 'required|string',
        ]);

        $user = ExternalUser::on('afiliados')
            ->where('username', $request->user)
            ->with(['memberships.membership'])
            ->first();

        if (!$user || sha1($request->password) !== $user->password) {
            return response()->json(['error' => 'Credenciales inválidas'], 401);
        }
        if ($user->memberships->isEmpty()) {
            return response()->json([
                'error' => 'Acceso restringido: No tienes una membresía activa'
            ], 403);
        }

        $membershipPrice = $user->memberships->first()->membership->price;

        $permissionData = match (true) {
            $membershipPrice >= 30 && $membershipPrice < 125 => ['permission' => 1, 'landings_allowed' => 1],
            $membershipPrice == 125 => ['permission' => 2, 'landings_allowed' => 3],
            $membershipPrice == 250 => ['permission' => 3, 'landings_allowed' => 5],
            $membershipPrice == 500 => ['permission' => 4, 'landings_allowed' => 10],
            $membershipPrice == 1000 => ['permission' => 5, 'landings_allowed' => 20],
            $membershipPrice == 2000 => ['permission' => 6, 'landings_allowed' => 50],
            default => ['permission' => 0, 'landings_allowed' => 0], // Si no cumple ningún caso (menor a 30)
        };

        // Validar que la membresía sea de al menos $30
        if ($permissionData < 1) {
            return response()->json([
                'error' => 'Acceso restringido: Necesitas una membresía de $30 o más'
            ], 403);
        }

        $token = $user->createToken('login_externalUser_token_' . $user->firstname, [], now()->addMinutes(config('sanctum.expiration')))->plainTextToken;

        return response()->json([
            'user' => $user,
            'token' => $token,
            'token_type' => 'Bearer',
            'permission' => $permissionData['permission'],
            'lamding_number' => $permissionData['landings_allowed'],
        ]);
    }


    public function logoutExternalUser(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Sesión cerrada correctamente'
        ]);
    }

    public function phothoprofile(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'image' => 'required|string',
                'external_user_id' => 'required|integer'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->image;
            $external_user_id = $request->external_user_id;

            if (!preg_match('/^data:image\/(\w+);base64,/', $data, $type)) {
                throw new \Exception('Formato de imagen no válido');
            }

            $image = substr($data, strpos($data, ',') + 1);
            $image = str_replace(' ', '+', $image);
            $imageType = strtolower($type[1]); // jpg, png, gif

            if (!in_array($imageType, ['jpg', 'jpeg', 'png', 'gif'])) {
                throw new \Exception('Tipo de imagen no soportado');
            }

            $imageName = "{$external_user_id}_" . Str::random(10) . ".{$imageType}";
            $storagePath = "profiles/{$imageName}";

            Storage::disk('public')->put($storagePath, base64_decode($image));

            return response()->json([
                'success' => true,
                'path' => Storage::url($storagePath),
                'filename' => $imageName,
                'message' => 'imagen almacenada correctamente'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al procesar la imagen',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}