<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Eventos;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Storage;

class EventosController extends Controller
{
    public function listEvents(Request $request)
    {
        $events = Eventos::paginate($request->input('per_page', 10));

        if ($events->isEmpty()) {
            return response()->json([
                'message' => 'No se encontraron eventos',
            ], Response::HTTP_NOT_FOUND);
        }

        // Procesar cada evento para agregar URL y Base64 de la imagen
        $events->getCollection()->transform(function ($event) {
            return $this->processEventImage($event);
        });

        return response()->json([
            'message' => 'Eventos encontrados',
            'data' => $events,
        ], Response::HTTP_OK);
    }

    public function registerEvents(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title_events' => 'required|max:255',
            'decriptions_events' => 'required|max:355',
            'image_events' => 'required|string',
            'date_events' => 'required|date_format:Y-m-d H:i:s',
            'duration_events' => 'required',
            'url_events' => 'required|max:500',
            'status_events' => 'required|max:50',
            'id_admins' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al crear el evento',
                    'errors' => $validator->errors()
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Procesar la imagen
        $imagePath = null;
        $imageName = null;
        $base64Image = $request->image_events;

        if ($request->has('image_events') && !empty($request->image_events)) {
            $imageBase64 = $request->image_events;

            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                $imageType = $type[1];
                $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);
                $imageData = base64_decode($imageBase64);

                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                $imageName = uniqid() . '.' . $imageType;
                $imagePath = 'images/events/' . $imageName;
                Storage::disk('public')->put($imagePath, $imageData);
            } else {
                return response()->json(
                    ['message' => 'El formato de la imagen no es válido'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        $evento = Eventos::create([
            'title_events' => $request->title_events,
            'decriptions_events' => $request->decriptions_events,
            'name_img_events' => $imageName,
            'image_events' => $imagePath,
            'base64_image' => $base64Image,
            'date_events' => $request->date_events,
            'duration_events' => $request->duration_events,
            'url_events' => $request->url_events,
            'status_events' => $request->status_events,
            'id_admins' => $request->id_admins
        ]);

        if (!$evento) {
            return response()->json(
                ['message' => 'Error al crear el evento'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        $newImageName = $evento->id . '_' . $imageName;
        $newImagePath = 'images/events/' . $newImageName;

        if ($imagePath && Storage::disk('public')->exists($imagePath)) {
            Storage::disk('public')->move($imagePath, $newImagePath);
            $evento->update([
                'name_img_events' => $newImageName,
                'image_events' => $newImagePath,
            ]);
        }

        $evento = $this->processEventImage($evento);

        return response()->json([
            'message' => 'Evento creado correctamente',
            'data' => $evento
        ], Response::HTTP_CREATED);
    }

    public function showEvents($id_event)
    {
        $event = Eventos::find($id_event);

        if (!$event) {
            return response()->json([
                'message' => 'Evento no encontrado',
            ], Response::HTTP_NOT_FOUND);
        }

        // Procesar el evento para la respuesta
        $event = $this->processEventImage($event);

        return response()->json([
            'message' => 'Evento encontrado',
            'data' => $event,
        ], Response::HTTP_OK);
    }

    public function upgradePartialEvents(Request $request, $id_event)
    {
        $evento = Eventos::find($id_event);
        if (!$evento) {
            return response()->json(
                ['message' => 'No se encontró el evento'],
                Response::HTTP_NOT_FOUND
            );
        }

        $validator = Validator::make($request->all(), [
            'title_events' => 'sometimes|max:255',
            'decriptions_events' => 'sometimes|max:355',
            'image_events' => 'nullable|string',
            'date_events' => 'sometimes|date_format:Y-m-d H:i:s',
            'duration_events' => 'sometimes',
            'url_events' => 'sometimes|max:500',
            'status_events' => 'sometimes|max:50',
            'id_admins' => 'sometimes'
        ]);

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el evento',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Manejo de la imagen
        if ($request->has('image_events') && !empty($request->image_events)) {
            $imageBase64 = $request->image_events;
            $evento->base64_image = $imageBase64;

            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);
                $imageData = base64_decode($imageBase64);

                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                // Eliminar imagen anterior si existe
                if ($evento->image_events && Storage::disk('public')->exists($evento->image_events)) {
                    Storage::disk('public')->delete($evento->image_events);
                }

                $imageName = $evento->id . '_' . uniqid() . '.' . $type[1];
                $imagePath = 'images/events/' . $imageName;
                Storage::disk('public')->put($imagePath, $imageData);

                $evento->image_events = $imagePath;
                $evento->name_img_events = $imageName;
            } else {
                return response()->json(
                    ['message' => 'La imagen no está en formato Base64'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        $evento->fill($request->only([
            'title_events',
            'decriptions_events',
            'date_events',
            'duration_events',
            'url_events',
            'status_events',
            'id_admins'
        ]));

        if (!$evento->save()) {
            return response()->json(
                ['message' => 'Error al actualizar el evento'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        $evento = $this->processEventImage($evento);

        return response()->json(
            [
                'message' => 'Evento actualizado correctamente',
                'data' => $evento
            ],
            Response::HTTP_OK
        );
    }
    
    private function processEventImage($event)
    {
        // Agregar la URL completa de la imagen
        $event->image_url = $event->image_events ? url(Storage::url($event->image_events)) : null;

        // Si no hay Base64 guardado pero hay imagen en storage, generarlo
        if (empty($event->base64_image) && $event->image_events && Storage::disk('public')->exists($event->image_events)) {
            $imageData = Storage::disk('public')->get($event->image_events);
            $event->base64_image = 'data:image/' . pathinfo($event->image_events, PATHINFO_EXTENSION) . ';base64,' . base64_encode($imageData);
        }

        return $event;
    }
}
