<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\ExternalUser;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class CourseController extends Controller
{
    public function listCourse(Request $request)
    {

        $courses = Course::paginate($request->input('per_page', 10));

        if ($courses->isEmpty()) {
            return response()->json([
                'message' => 'No se encontraron cursos',
            ], Response::HTTP_NOT_FOUND);
        }

        $courses->getCollection()->transform(function ($course) {
            return $this->processCourseImage($course);
        });


        return response()->json([
            'message' => 'Cursos encontrados',
            'data' => $courses,
        ], Response::HTTP_OK);
    }

    public function registerCourse(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'nombre_courses' => 'required|max:100',
                'status_courses' => 'required|max:50',
                'image_courses' => 'required|string',
                'descriptions_courses' => 'required|string',
                'id_category' => 'required',
                'url_video' => 'required',
                'id_career' => 'required'
            ]
        );

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar el curso',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Procesar la imagen
        $imagePath = null;
        $imageName = null;
        $base64Image = null;

        if ($request->has('image_courses') && !empty($request->image_courses)) {
            $imageBase64 = $request->image_courses;
            $base64Image = $imageBase64; // Guardar el Base64 original

            // Verificar si la imagen es Base64 válida
            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                $imageType = $type[1];
                $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);
                $imageData = base64_decode($imageBase64);

                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                // Generar un nombre único para la imagen
                $imageName = uniqid() . '.' . $imageType;
                $imagePath = 'images/courses/' . $imageName;
                Storage::disk('public')->put($imagePath, $imageData);
            } else {
                return response()->json(
                    ['message' => 'El formato de la imagen no es válido'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        $course = Course::create([
            'nombre_courses' => $request->nombre_courses,
            'status_courses' => $request->status_courses,
            'image_courses' => $imagePath,
            'name_img_courses' => $imageName,
            'descriptions_courses' => $request->descriptions_courses,
            'id_category' => $request->id_category,
            'url_video' => $request->url_video,
            'id_career' => $request->id_career
        ]);

        if (!$course) {
            return response()->json(
                ['message' => 'Error al guardar el dato'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }
        $course = $this->processCourseImage($course);

        return response()->json(
            [
                'message' => 'El curso se registró correctamente',
                'data' => $course
            ],
            Response::HTTP_CREATED
        );
    }

    public function upgradePartialCourse(Request $request, $id_courses)
    {
        $course = Course::find($id_courses);
        if (!$course) {
            return response()->json(
                ['message' => 'No se encontró el curso'],
                Response::HTTP_NOT_FOUND
            );
        }

        $validator = Validator::make($request->all(), [
            'nombre_courses' => 'sometimes|max:100',
            'status_courses' => 'sometimes|max:50',
            'image_courses' => 'sometimes|string',
            'descriptions_courses' => 'sometimes|string',
            'id_category' => 'sometimes',
            'url_video' => 'sometimes',
            'id_career' => 'sometimes'
        ]);

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el curso',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        $updatedFields = 0;

        // Manejo de la imagen (solo si se envía y no está vacía)
        if ($request->filled('image_courses')) {
            $imageBase64 = $request->image_courses;

            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                $imageData = base64_decode(substr($imageBase64, strpos($imageBase64, ',') + 1));

                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                $imageName = uniqid().'.'.$type[1];
                $imagePath = 'images/courses/'.$imageName;

                // Eliminar imagen anterior si existe
                if ($course->image_courses && Storage::disk('public')->exists($course->image_courses)) {
                    Storage::disk('public')->delete($course->image_courses);
                }

                // Guardar nueva imagen
                Storage::disk('public')->put($imagePath, $imageData);

                // Actualizar solo campos existentes
                $course->image_courses = $imagePath;
                $course->name_img_courses = $imageName;

                $updatedFields++;
            } else {
                return response()->json(
                    ['message' => 'La imagen no está en formato Base64 válido'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        $fieldsToUpdate = [
            'nombre_courses',
            'status_courses',
            'descriptions_courses',
            'id_category',
            'url_video',
            'id_career'
        ];

        foreach ($fieldsToUpdate as $field) {
            if ($request->has($field)) {
                $course->$field = $request->$field !== null ? $request->$field : null;
                $updatedFields++;
            }
        }

        if ($updatedFields === 0) {
            return response()->json(
                ['message' => 'No se recibieron campos para actualizar'],
                Response::HTTP_OK
            );
        }

        if (!$course->save()) {
            return response()->json(
                ['message' => 'Error al guardar los cambios en la base de datos'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        $course = $this->processCourseImage($course);

        return response()->json(
            [
                'message' => 'Curso actualizado correctamente',
                'updated_fields' => $updatedFields,
                'data' => $course
            ],
            Response::HTTP_OK
        );
    }

    public function showCourse($id_courses)
    {
        $course = Course::find($id_courses);

        if (!$course) {
            return response()->json([
                'message' => 'Curso no encontrado',
            ], Response::HTTP_NOT_FOUND);
        }

        $course = $this->processCourseImage($course);

        return response()->json([
            'message' => 'Curso encontrado',
            'data' => $course,
        ], Response::HTTP_OK);
    }

    public function searchCourseWithModules($id_courses)
    {
        $course = Course::find($id_courses);

        if (!$course) {
            return response()->json([
                'message' => 'Curso no encontrado',
            ], Response::HTTP_NOT_FOUND);
        }

        $course = $this->processCourseImage($course);

        return response()->json([
            'course' => $course,
            'modules' => $course->modules,
        ]);
    }

    public function getCourseUsers($id_courses)
    {
        try {
            // Obtener el curso con usuarios externos relacionados
            $course = Course::with('externalUsers')->findOrFail($id_courses);


            $course = $this->processCourseImage($course);

            return response()->json([
                'success' => true,
                'course' => [
                    'id' => $course->id_courses,
                    'name' => $course->nombre_courses,
                    'description' => $course->descriptions_courses,
                    'image' => $course->image
                ],
                'external_users' => $course->externalUsers->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'username' => $user->username,
                        'email' => $user->email,
                        'firstname' => $user->firstname,
                        'lastname' => $user->lastname
                    ];
                })
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener los usuarios del curso',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function processCourseImage($course)
    {
        if (!$course) {
            return $course;
        }

        // URL completa de la imagen
        $course->image_url = $course->image_course ? url(Storage::url($course->image_course)) : null;

        // Generar Base64 si no está guardado pero existe la imagen
        if (empty($course->base64_image) && $course->image_course && Storage::disk('public')->exists($course->image_user)) {
            $imageData = Storage::disk('public')->get($course->image_course);
            $course->base64_image = 'data:image/' . pathinfo($course->image_course, PATHINFO_EXTENSION) . ';base64,' . base64_encode($imageData);
        }

        return $course;
    }

    public function enrollUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'external_user_id' => 'required|integer',
            'id_courses' => 'required|integer'
        ], [
            'external_user_id.required' => 'El ID de usuario externo es obligatorio',
            'id_courses.required' => 'El ID de curso es obligatorio',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Usamos la conexión 'afiliados' para el usuario
            $externalUser = ExternalUser::on('afiliados')->findOrFail($request->external_user_id);

            // Usamos la conexión 'cursos' para el curso
            $course = Course::on('mysql')->findOrFail($request->id_courses);

            $currentTime = now();

            // Verificar inscripción usando el modelo pivot
            $isEnrolled = DB::connection('mysql')
                ->table('course_user')
                ->where('external_user_id', $externalUser->id)
                ->where('id_courses', $course->id_courses)
                ->exists();

            if ($isEnrolled) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ya estás inscrito en este curso'
                ], 409);
            }

            DB::connection('mysql')
                ->table('course_user')
                ->insert([
                    'external_user_id' => $externalUser->id,
                    'id_courses' => $course->id_courses,
                    'created_at' => $currentTime,
                    'updated_at' => $currentTime
                ]);

            return response()->json([
                'success' => true,
                'message' => "¡Felicidades {$externalUser->firstname} {$externalUser->lastname}! Has sido inscrito correctamente en el curso: {$course->nombre_courses}",
                'data' => [
                    'user' => [
                        'id' => $externalUser->id,
                        'name' => "{$externalUser->firstname} {$externalUser->lastname}"
                    ],
                    'course' => [
                        'id' => $course->id_courses,
                        'name' => $course->nombre_courses,
                        'description' => $course->descriptions_courses
                    ],
                    'enrollment_details' => [
                        'created_at' => $currentTime->toDateTimeString(),
                        'updated_at' => $currentTime->toDateTimeString()
                    ]
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al procesar la inscripción',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    public function checkEnrollment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'external_user_id' => 'required|integer',
            'id_courses' => 'required|integer'
        ], [
            'external_user_id.required' => 'El ID de usuario es obligatorio',
            'id_courses.required' => 'El ID de curso es obligatorio',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {

            $externalUser = ExternalUser::on('afiliados')->find($request->external_user_id);
            if (!$externalUser) {
                return response()->json([
                    'success' => false,
                    'message' => 'Usuario no encontrado'
                ], 404);
            }


            $course = Course::on('mysql')->find($request->id_courses);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Curso no encontrado'
                ], 404);
            }


            $isEnrolled = DB::connection('mysql')
                ->table('course_user')
                ->where('external_user_id', $request->external_user_id)
                ->where('id_courses', $request->id_courses)
                ->exists();

            return response()->json([
                'success' => true,
                'is_enrolled' => $isEnrolled,
                'message' => $isEnrolled
                    ? 'El usuario está inscrito en este curso'
                    : 'El usuario no está inscrito en este curso',
                'data' => [
                    'user' => [
                        'id' => $externalUser->id,
                        'name' => "{$externalUser->firstname} {$externalUser->lastname}"
                    ],
                    'course' => [
                        'id' => $course->id_courses,
                        'name' => $course->nombre_courses
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al verificar la inscripción',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function countCourse(Request $request){
        $count = Course::count();

        return response()->json([
            'message' => 'Conteo de cursos',
            'total_cursos' => $count,
        ], Response::HTTP_OK);
    }

    }