<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;


class CountryController extends Controller
{
    public function listCountry(Request $request)
    {
        $perPage = $request->input('per_page', 20);

        $countries = Country::paginate($perPage);

        if ($countries->isEmpty()) {

            return response()->json(
                ['message' => 'No se encontraron los países'],
                Response::HTTP_NOT_FOUND
            );
        }

        return response()->json(
            $countries,
            Response::HTTP_OK
        );
    }

    //revisar si es necesaria la funcion registerCountry
    public function registerCountry(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [

                'iso_paises' => 'required|max:2',
                'nombre_paises' => 'required|max:80'
            ]
        );

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar el país',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        $countries = Country::create([
            'iso_paises' => $request->iso_paises,
            'nombre_paises' => $request->nombre_paises,
        ]);

        if (!$countries) {
            return response()->json(
                ['message' => 'Error al guardar el dato'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        return response()->json(
            ['message' => 'El país se registro correctamente',
            'data' => $countries],
            Response::HTTP_CREATED
        );
    }

    public function showCountry($id_paises)
    {
        $country = Country::find($id_paises);
        if (!$country) {

            return response()->json(
                ['message' => 'No se encontro el pais'],
                Response::HTTP_NOT_FOUND
            );
        }

            return response()->json(
                ['message' => $country],
                Response::HTTP_OK
            );
    }

    //revisar si es necesaria la funcion de update
    public function upgradePartialCountry(Request $request, $id_paises)
    {
        $country = Country::find($id_paises);
        if (!$country) {
            return response()->json(
                ['message' => 'No se encontró el país'],
                Response::HTTP_NOT_FOUND
            );
        }

        $validator = Validator::make($request->all(), [
            'iso_paises' => 'sometimes|max:2',
            'nombre_paises' => 'sometimes|max:80'
        ]);

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error de validación',
                    'errors' => $validator->errors()
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        $updatedFields = 0;
        
        if ($request->has('iso_paises')) {
            $country->iso_paises = $request->iso_paises;
            $updatedFields++;
        }

        if ($request->has('nombre_paises')) {
            $country->nombre_paises = $request->nombre_paises;
            $updatedFields++;
        }
        if ($updatedFields === 0) {
            return response()->json(
                ['message' => 'No se recibieron campos para actualizar'],
                Response::HTTP_OK
            );
        }

        if (!$country->save()) {
            return response()->json(
                ['message' => 'Error al guardar los cambios en la base de datos'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        return response()->json(
            [
                'message' => 'País actualizado correctamente',
                'updated_fields' => $updatedFields,
                'data' => $country
            ],
            Response::HTTP_OK
        );
    }
}
