<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class CategoryController extends Controller
{
    //Mostrar todo

    public function listCategory(Request $request)
    {

        $perPage = $request->input('per_page', 20);


        $categories = Category::paginate($perPage);

        if ($categories->isEmpty()) {
            return response()->json(['message' => 'No se encontraron categorías'], Response::HTTP_NOT_FOUND);
        }

        return response()->json($categories, Response::HTTP_OK);
    }

    public function registerCategory(Request $request)
    {
        $validator = Validator::make($request->all(), [

            'nombre_category' => 'required|max:35',
            'status_category' => 'required|max:50'

        ]);

        //Verifico si la validacion fallo
        if ($validator->fails()) {
            return response()->json(
                ['message' => 'Error al registrar categoria', 'errors' => $validator->errors()],
                Response::HTTP_BAD_REQUEST
            );
        }

        //Si la validacion fue correcto creo el campo

        $categories = Category::create([
            'nombre_category' => $request->nombre_category,
            'status_category' => $request->status_category

        ]);

        if (!$categories) {
            return response()->json(['message' => 'Error al registrar categoria'], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
        return response()->json(['message' => 'Categoria registrada correctamente',
        'data' => $categories], Response::HTTP_CREATED);
    }


    public function upgradePartialCategory(Request $request, $id_category){
        // Buscar la categoría por su ID
        $category = Category::find($id_category);
        if (!$category) {
            // Si no existe, retorna un mensaje de error
            return response()->json(
                ['message' => 'No se encontró la categoría'],
                Response::HTTP_NOT_FOUND
            );
        }

        // Validar los datos a actualizar
        $validator = Validator::make($request->all(), [
            'nombre_category' => 'sometimes|max:35', // "sometimes" permite que el campo sea opcional
            'status_category' => 'sometimes|max:50', // "sometimes" permite que el campo sea opcional
        ]);

        // Si los datos son inválidos, mensaje de error
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar la categoría',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Validar campos vacíos
        $errors = [];
        $fields = ['nombre_category', 'status_category'];

        foreach ($fields as $field) {
            if ($request->has($field) && empty($request->$field)) {
                $errors[$field] = 'El campo ' . $field . ' no puede estar vacío.';
            }
        }

        if (!empty($errors)) {
            return response()->json(
                [
                    'message' => 'Error al actualizar la categoría: campos vacíos',
                    'errors' => $errors,
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Actualizar solo los campos proporcionados
        $dataToUpdate = array_filter($request->only([
            'nombre_category',
            'status_category',
        ]), function ($value) {
            return !empty($value); // Filtra campos vacíos
        });

        $category->fill($dataToUpdate);
        $category->save();

        // Mensaje de actualización realizada
        return response()->json(
            [
                'message' => 'Categoría actualizada correctamente',
                'data' => $category,
            ],
            Response::HTTP_OK
        );
    }

    public function showCategory($id_category)
    {
        $category = Category::find($id_category);
        if (!$category) {
            //si no existe retorna un mensaje de error
            return response()->json(
                ['message' => 'No se encontro la categoria'],
                Response::HTTP_NOT_FOUND
            );
        }

        //si existe retorna un mensaje con los datos solicitados
        return response()->json(
            ['message' => $category],
            Response::HTTP_OK
        );
    }

    public function getCategoryWithCourses($id_category)
    {
        // Forzar consulta sin scopes
        $category = Category::with(['courses' => function($query) {
            $query->withoutGlobalScopes();
        }])->find($id_category);

        if (!$category) {
            return response()->json(['message' => 'Categoría no encontrada'], 404);
        }

        return response()->json([
            'message' => 'Datos encontrados',
            'data' => $category
        ]);
    }
    }



