<?php

namespace App\Http\Controllers\Api;

use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use App\Models\Admin;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class AdminController extends Controller
{
   public function listAdmin(Request $request)
{

    $perPage = $request->input('per_page', 20);


    $admins = Admin::paginate($perPage);

    if ($admins->isEmpty()) {

        return response()->json(
            ['message' => 'No se encontró el admin'],
            Response::HTTP_NOT_FOUND
        );
    }


    return response()->json(
        $admins,
        Response::HTTP_OK
    );
}

    public function RegisterAdmins(Request $request)
    {
        $validator = Validator::make(
            $request->all(),
            [
                'nombre_admins' => 'required|max:100',
                'apellido_admins' => 'required|max:100',
                'ci_admins' => 'required|max:30|unique:admins_life',
                'correo_admins' => 'required|max:100|email|unique:admins_life',
                'password_admins' => 'required|max:61',
                'status_admins' => 'required|max:50',
                'id_paises' => 'required',
                'id_permissions' => 'required'
            ]
        );

        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar el admin',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        $admin = Admin::create([
            'nombre_admins' => $request->nombre_admins,
            'apellido_admins' => $request->apellido_admins,
            'ci_admins' => $request->ci_admins,
            'correo_admins' => $request->correo_admins,
            'password_admins' => Hash::make($request->password_admins),
            'status_admins' => $request->status_admins,
            'id_paises' => $request->id_paises,
            'id_permissions' => $request->id_permissions
        ]);

        if (!$admin) {
            return response()->json(
                ['message' => 'Error al guardar los datos'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        // $token = $admin->createToken('register_admin_token')->plainTextToken;
        return response()->json(
            [
                'message' => 'El usuario fue registrado correctamente',
                'data' => $admin,
            ],
            Response::HTTP_CREATED
        );
    }

    public function ShowAdmins($id_admins)
    {
        $admin = Admin::find($id_admins);
        if (!$admin) {

            return response()->json(
                ['message' => 'No se encontro el admin'],
                Response::HTTP_NOT_FOUND
            );
        }

        return response()->json(
            ['message' => $admin],
            Response::HTTP_OK
        );
    }

    public function UpgradePartialAdmins(Request $request, $id_admins)
    {

        $admin = Admin::find($id_admins);
        if (!$admin) {

            return response()->json(
                ['message' => 'No se encontro el usuario'],
                Response::HTTP_NOT_FOUND
            );
        }

        $validator = Validator::make($request->all(), [
            'nombre_admins' => 'required|max:100',
            'apellido_admins' => 'required|max:100',
            'ci_admins' => 'required|max:30',
            'correo_admins' => 'required|max:100|email|unique:admins_life',
            'password_admins' => 'required|max:61',
            'status_admins' => 'required|max:50',
            'id_paises' => 'required',
            'id_permissions' => 'required'
        ]);


        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el admin',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }


        if ($request->has('nombre_admins')) {
            $admin->nombre_admins = $request->nombre_admins;
        }

        if ($request->has('apellido_admins')) {
            $admin->apellido_admins = $request->apellido_admins;
        }

        if ($request->has('ci_admins')) {
            $admin->ci_user = $request->ci_user;
        }

        if ($request->has('correo_admins')) {
            $admin->correo_admins = $request->correo_admins;
        }

        if ($request->has('password_admins')) {
            $admin->password_admins = $request->password_admins;
        }

        if ($request->has('status_admins')) {
            $admin->status_admins = $request->status_admins;
        }

        if ($request->has('id_paises')) {
            $admin->id_paises = $request->id_paises;
        }

        if ($request->has('id_permissions')) {
            $admin->id_permissions = $request->id_permissions;
        }

        $admin->save();

        return response()->json(
            ['message' => $admin],
            Response::HTTP_OK
        );
    }

    public function validateLoginAdmin(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'correo_admins' => 'required|email',
            'password_admins' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validación fallida',
                'errors' => $validator->errors()
            ], Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $admin = Admin::where('correo_admins', $request->correo_admins)->first();

        if (!$admin || !Hash::check($request->password_admins, $admin->password_admins)) {
            return response()->json([
                'success' => false,
                'message' => 'Credenciales incorrectas'
            ], Response::HTTP_UNAUTHORIZED);
        }


        $token = $admin->createToken('login_admin_token' . $admin->nombre_admins, [], now()->addMinutes(config('sanctum.expiration')))->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Bienvenido ' . $admin->nombre_admins,
            'data' => $admin,
            'access_token' => $token,
            'token_type' => 'Bearer',
            'expires_in' => config('sanctum.expiration') * 60,
        ], Response::HTTP_OK);
    }

    public function logoutAdmin(Request $request)
    {
        if (!Auth::check()) {
            return response()->json(['message' => 'No estás autenticado.'], Response::HTTP_UNAUTHORIZED);
        }

        $token = $request->user()->currentAccessToken();

        if ($token) {

            $token->delete();
            return response()->json(['message' => 'Logout exitoso.'], Response::HTTP_OK);
        }

        return response()->json(['message' => 'No se pudo cerrar la sesión.'], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function getCurrentAdmin(Request $request)
{
    return response()->json([
        'admin' => $request->user() 
    ]);
}
}
