<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Videomenu;
use App\Models\ExternalUser;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class VideomenuController extends Controller
{
    public function listVideomenu($external_user_id)
    {
        
            try {
                $externaluser = ExternalUser::on('afiliados')->select([
                    'id',
                    'username',
                    'firstname',
                    'lastname',
                    'phone'
                ])
                ->findOrFail($external_user_id);
                
                return response()->json([
                    'success' => true,
                    'user' => $externaluser
                ]);
                
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Usuario no encontrado'
                ], 404);
            }
        }


    public function registerVideomenu(Request $request)
    {

        $validator = Validator::make(
            $request->all(),
            [
                'nombre_videomenu' => 'required|max:100',
                'description_videomenu' => 'required|max:300',
                'url_videomenu' => 'required|max:500',
                'url_menu' => 'required|max:500',
                'type_videomenu' => 'required|max:50'
            ]
        );


        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar el video',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }



        $videomenu = Videomenu::create([

            'nombre_video' => $request->nombre_video,
            'description_video' => $request->description_video,
            'url_video' => $request->url_video,
            'status_video' => $request->status_video,

        ]);


        if (!$videomenu) {
            return response()->json(
                ['message' => 'Error al guardar los datos'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }


        return response()->json(
            [
                'message' => 'El video se registro correctamente',
                'data' => $videomenu
            ],
            Response::HTTP_CREATED
        );
    }


    public function showVideomenu($id_videomenu)

    {
        $videomenu = Videomenu::find($id_videomenu);
        if (!$videomenu) {


            return response()->json(
                ['message' => 'No se encontro el video'],
                Response::HTTP_NOT_FOUND
            );
        }


        return response()->json(
            ['message' => $videomenu],
            Response::HTTP_OK
        );
    }


    public function upgradePartialVideomenu(Request $request, $id_videomenu)
    {
        // Buscar el video por su ID
        $videomenu = Videomenu::find($id_videomenu);
        if (!$videomenu) {
            // Si no existe, retorna un mensaje de error
            return response()->json(
                ['message' => 'No se encontró el video'],
                Response::HTTP_NOT_FOUND
            );
        }

        // Validar los datos a actualizar
        $validator = Validator::make($request->all(), [
            'nombre_videomenu' => 'sometimes|required|max:100', // "sometimes" permite que el campo sea opcional
            'description_videomenu' => 'sometimes|required|max:300',
            'url_videomenu' => 'sometimes|required|max:500',
            'url_menu' => 'sometimes|required|max:500',
            'type_videomenu' => 'sometimes|required|max:50',
        ]);

        // Si los datos son inválidos, mensaje de error
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el video',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Validar campos vacíos
        $errors = [];
        $fields = [
            'nombre_videomenu',
            'description_videomenu',
            'url_videomenu',
            'url_menu',
            'type_videomenu',
        ];

        foreach ($fields as $field) {
            if ($request->has($field)) {
                if (empty($request->$field)) {
                    $errors[$field] = 'El campo ' . $field . ' no puede estar vacío.';
                }
            }
        }

        if (!empty($errors)) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el video: campos vacíos',
                    'errors' => $errors,
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Actualizar solo los campos proporcionados
        $dataToUpdate = array_filter($request->only([
            'nombre_videomenu',
            'description_videomenu',
            'url_videomenu',
            'url_menu',
            'type_videomenu',
        ]), function ($value) {
            return !empty($value); // Filtra campos vacíos
        });

        $videomenu->fill($dataToUpdate);
        $videomenu->save();

        // Mensaje de actualización realizada
        return response()->json(
            [
                'message' => 'Video actualizado correctamente',
                'data' => $videomenu,
            ],
            Response::HTTP_OK
        );
    }
}
