<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Storage;


class VideoController extends Controller
{

    public function listVideo(Request $resquest)
    {
        $perPage = $resquest->input('per_page', 20);

        $video = Video::paginate($perPage);


        if ($video->isEmpty()) {

            return response()->json(
                ['message' => 'No se encontro el video'],
                Response::HTTP_NOT_FOUND
            );
        }

        return response()->json(
            $video,
            Response::HTTP_OK
        );
    }


    public function registerVideo(Request $request)
    {

        $validator = Validator::make(
            $request->all(),
            [
                'nombre_video' => 'required|max:100',
                'description_video' => 'required|string',
                'image_video' => 'string|',
                'url_video' => 'required|max:500',
                'status_video' => 'required|max:50'
            ]
        );


        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar el video',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Procesar la imagen
        $imagePath = null;
        $imageName = null;

        if ($request->has('image_video') && !empty($request->image_video)) {
            $imageBase64 = $request->image_video;

            // Verificar si la imagen es Base64 válida
            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                // Extraer el tipo de imagen (png, jpeg, etc.)
                $imageType = $type[1];

                // Eliminar el prefijo "data:image/<tipo>;base64,"
                $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);

                // Decodificar la imagen Base64
                $imageData = base64_decode($imageBase64);

                // Verificar si la decodificación fue exitosa
                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                // Generar un nombre único para la imagen
                $imageName = uniqid() . '.' . $imageType;

                // Guardar la imagen en el storage
                $imagePath = 'images/' . $imageName;
                Storage::disk('public')->put($imagePath, $imageData);
            } else {
                return response()->json(
                    ['message' => 'El formato de la imagen no es válido'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        $video = Video::create([

            'nombre_video' => $request->nombre_video,
            'description_video' => $request->description_video,
            'name_img_video' => $imageName,
            'image_video' => $imagePath,
            'url_video' => $request->url_video,
            'status_video' => $request->status_video,

        ]);


        if (!$video) {
            return response()->json(
                ['message' => 'Error al guardar el video'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }


        return response()->json(
            ['message' => 'Video registrado correctamente', 'data' => $video],
            Response::HTTP_OK
        );
    }


    public function showVideo($id_video)

    {
        $video = Video::find($id_video);
        if (!$video) {


            return response()->json(
                ['message' => 'No se encontro el video'],
                Response::HTTP_NOT_FOUND
            );
        }


        return response()->json(
            ['message' => $video],
            Response::HTTP_OK
        );
    }


    public function upgradePartialVideo(Request $request, $id_video)
    {
        // Buscar el video
        $video = Video::find($id_video);
        if (!$video) {
            return response()->json(
                ['message' => 'No se encontró el video'],
                Response::HTTP_NOT_FOUND
            );
        }

        // Reglas de validación (campos opcionales)
        $validator = Validator::make($request->all(), [
            'nombre_video' => 'sometimes|max:100',
            'description_video' => 'sometimes|max:100',
            'name_img_video' => 'sometimes|string|max:255',
            'image_video' => 'nullable|string', // Cambiar a nullable para permitir Base64
            'url_video' => 'sometimes|max:500',
            'status_video' => 'sometimes|max:50'
        ]);

        // Si la validación falla, retornar errores
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el video',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Validar campos vacíos
        $errors = [];
        $fields = [
            'nombre_video',
            'description_video',
            'name_img_video',
            'image_video',
            'url_video',
            'status_video'
        ];

        foreach ($fields as $field) {
            if ($request->has($field) && empty($request->$field)) {
                $errors[$field] = 'El campo ' . $field . ' no puede estar vacío.';
            }
        }

        // Si hay campos vacíos, retornar errores
        if (!empty($errors)) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el video: campos vacíos',
                    'errors' => $errors,
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Manejo de la imagen
        if ($request->has('image_video') && !empty($request->image_video)) {
            $imageBase64 = $request->image_video;

            // Verificar si la imagen es Base64
            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);
                $imageData = base64_decode($imageBase64);

                // Comprobar si la decodificación fue exitosa
                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                // Generar un nombre único para la imagen
                $imageName = uniqid() . '.' . $type[1]; // Usar la extensión extraída
                $imagePath = 'images/' . $imageName;

                // Guardar la imagen en el storage
                Storage::disk('public')->put($imagePath, $imageData);

                // Actualizar el video con la nueva imagen y nombre de la imagen
                $video->image_video = $imagePath;
                $video->name_img_video = $imageName;
            } else {
                return response()->json(
                    ['message' => 'La imagen no está en formato Base64'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        // Filtrar campos vacíos
        $dataToUpdate = array_filter($request->only([
            'nombre_video',
            'description_video',
            'url_video',
            'status_video'
        ]), function ($value) {
            return !empty($value); // Solo se incluyen valores no vacíos
        });

        // Actualizar solo los campos que estén presentes y no estén vacíos
        $video->fill($dataToUpdate);

        // Guardar los cambios
        $video->save();

        return response()->json(
            ['message' => 'Video actualizado correctamente', 'data' => $video],
            Response::HTTP_OK
        );
    }
}
