<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;


class CountryController extends Controller
{
    public function listCountry(Request $request)
    {

        $perPage = $request->input('per_page', 20);

        $countries = Country::paginate($perPage);

        if ($countries->isEmpty()) {

            return response()->json(
                ['message' => 'No se encontraron los países'],
                Response::HTTP_NOT_FOUND
            );
        }

        return response()->json(
            $countries,
            Response::HTTP_OK
        );
    }

    //Controlar el registro
    public function registerCountry(Request $request)
    {
        //Valida los datos a registrar
        $validator = Validator::make(
            $request->all(),
            [

                'iso_paises' => 'required|max:2',
                'nombre_paises' => 'required|max:80'
            ]
        );

        //Si el registro falla
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar el país',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        //Si la validacion es correcta creo

        $countries = Country::create([
            'iso_paises' => $request->iso_paises,
            'nombre_paises' => $request->nombre_paises,
        ]);

        //Si el registro no es exitoso
        if (!$countries) {
            return response()->json(
                ['message' => 'Error al guardar el dato'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        //Si el registro es exitoso
        return response()->json(
            ['message' => 'El país se registro correctamente',
            'data' => $countries],
            Response::HTTP_CREATED
        );
    }

    //Busqueda por id
    public function showCountry($id_paises)
    {
        $country = Country::find($id_paises);
        if (!$country) {
        //si no existe retorna un mensaje de error
            return response()->json(
                ['message' => 'No se encontro el pais'],
                Response::HTTP_NOT_FOUND
            );
        }

        //si existe retorna un mensaje con los datos solicitados
            return response()->json(
                ['message' => $country],
                Response::HTTP_OK
            );
    }

    //Actualizacion de datos
    public function upgradePartialCountry(Request $request, $id_paises)
    {
        // Buscar el país por su ID
        $country = Country::find($id_paises);
        if (!$country) {
            return response()->json(
                ['message' => 'No se encontró el país'],
                Response::HTTP_NOT_FOUND
            );
        }

        // Validar los datos a actualizar (campos opcionales)
        $validator = Validator::make($request->all(), [
            'iso_paises' => 'sometimes|max:2',
            'nombre_paises' => 'sometimes|max:80'
        ]);

        // Si los datos son inválidos, retornar mensaje de error
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error de validación',
                    'errors' => $validator->errors()
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        $updatedFields = 0;

        // Actualizar solo los campos proporcionados
        if ($request->has('iso_paises')) {
            $country->iso_paises = $request->iso_paises;
            $updatedFields++;
        }

        if ($request->has('nombre_paises')) {
            $country->nombre_paises = $request->nombre_paises;
            $updatedFields++;
        }

        // Verificar si hubo campos para actualizar
        if ($updatedFields === 0) {
            return response()->json(
                ['message' => 'No se recibieron campos para actualizar'],
                Response::HTTP_OK
            );
        }

        // Guardar los cambios
        if (!$country->save()) {
            return response()->json(
                ['message' => 'Error al guardar los cambios en la base de datos'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        // Respuesta exitosa
        return response()->json(
            [
                'message' => 'País actualizado correctamente',
                'updated_fields' => $updatedFields,
                'data' => $country
            ],
            Response::HTTP_OK
        );
    }
}
