<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Career;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Storage;

class CareerController extends Controller
{
    // Mostrar todo con paginación
    public function listCareer(Request $request)
    {
        // Obtener el número de registros por página (por defecto 10)
        $perPage = $request->input('per_page', 10);

        // Obtener todas las carreras con paginación
        $careers = Career::paginate($perPage);

        // Verificar si hay carreras
        if ($careers->isEmpty()) {
            return response()->json([
                'message' => 'No se encontraron carreras',
            ], Response::HTTP_NOT_FOUND);
        }

        // Procesar cada carrera para agregar la URL de la imagen y Base64
        $careers->getCollection()->transform(function ($career) {
            return $this->processCareerImage($career);
        });

        // Devolver la respuesta con todas las carreras
        return response()->json([
            'message' => 'Carreras encontradas',
            'data' => $careers,
        ], Response::HTTP_OK);
    }

    // Registrar carrera
    public function registerCareer(Request $request)
    {
        // Validar los datos a registrar
        $validator = Validator::make(
            $request->all(),
            [
                'nombre_career' => 'required|max:100',
                'descripcion_career' => 'required|max:300',
                'status_career' => 'required|max:50',
                'image_career' => 'required|string', // Base64 string
            ]
        );

        // Si la validación falla, retornar errores
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar la carrera',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Procesar la imagen
        $imagePath = null;
        $imageName = null;
        $base64Image = null;

        if ($request->has('image_career') && !empty($request->image_career)) {
            $imageBase64 = $request->image_career;
            $base64Image = $imageBase64; // Guardar el Base64 original

            // Verificar si la imagen es Base64 válida
            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                // Extraer el tipo de imagen (png, jpeg, etc.)
                $imageType = $type[1];

                // Eliminar el prefijo "data:image/<tipo>;base64,"
                $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);

                // Decodificar la imagen Base64
                $imageData = base64_decode($imageBase64);

                // Verificar si la decodificación fue exitosa
                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                // Generar un nombre único para la imagen
                $imageName = uniqid() . '.' . $imageType;

                // Guardar la imagen en el storage
                $imagePath = 'images/careers/' . $imageName;
                Storage::disk('public')->put($imagePath, $imageData);
            } else {
                return response()->json(
                    ['message' => 'El formato de la imagen no es válido'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        // Crear la carrera
        $career = Career::create([
            'nombre_career' => $request->nombre_career,
            'descripcion_career' => $request->descripcion_career,
            'status_career' => $request->status_career,
            'name_img_career' => $imageName,
            'image_career' => $imagePath,
            'base64_image' => $base64Image, // Guardar el Base64 original
        ]);

        // Si el registro no es exitoso
        if (!$career) {
            return response()->json(
                ['message' => 'Error al guardar el dato'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        // Procesar la carrera para la respuesta
        $career = $this->processCareerImage($career);

        // Si el registro es exitoso
        return response()->json(
            [
                'message' => 'La carrera se registró correctamente',
                'data' => $career
            ],
            Response::HTTP_CREATED
        );
    }

    // Actualización parcial
    public function upgradePartialCareer(Request $request, $id_career)
    {
        // Busqueda por id
        $career = Career::find($id_career);
        if (!$career) {
            // Si no existe, retorna un mensaje de error
            return response()->json(
                ['message' => 'No se encontró la carrera'],
                Response::HTTP_NOT_FOUND
            );
        }

        // Valido los datos a actualizar
        $validator = Validator::make($request->all(), [
            'nombre_career' => 'sometimes|max:100',
            'descripcion_career' => 'sometimes|max:300',
            'status_career' => 'sometimes|max:50',
            'image_career' => 'nullable|string', // Base64 string
        ]);

        // Si los datos son inválidos, mensaje de error
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar la carrera',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // Manejo de la imagen
        if ($request->has('image_career') && !empty($request->image_career)) {
            $imageBase64 = $request->image_career;
            
            // Guardar el Base64 original
            $career->base64_image = $imageBase64;

            // Verificar si la imagen es Base64
            if (preg_match('/^data:image\/(\w+);base64,/', $imageBase64, $type)) {
                $imageBase64 = substr($imageBase64, strpos($imageBase64, ',') + 1);
                $imageData = base64_decode($imageBase64);

                // Comprobar si la decodificación fue exitosa
                if ($imageData === false) {
                    return response()->json(
                        ['message' => 'La imagen Base64 no es válida'],
                        Response::HTTP_BAD_REQUEST
                    );
                }

                // Generar un nombre único para la imagen
                $imageName = uniqid() . '.' . $type[1];
                $imagePath = 'images/careers/' . $imageName;

                // Eliminar la imagen anterior si existe
                if ($career->image_career && Storage::disk('public')->exists($career->image_career)) {
                    Storage::disk('public')->delete($career->image_career);
                }

                // Guardar la nueva imagen en el storage
                Storage::disk('public')->put($imagePath, $imageData);

                // Actualizar la carrera con la nueva imagen
                $career->image_career = $imagePath;
                $career->name_img_career = $imageName;
            } else {
                return response()->json(
                    ['message' => 'La imagen no está en formato Base64'],
                    Response::HTTP_BAD_REQUEST
                );
            }
        }

        // Actualizar solo los campos proporcionados
        $career->fill($request->only([
            'nombre_career',
            'descripcion_career',
            'status_career',
        ]));

        // Guardar los cambios
        if (!$career->save()) {
            return response()->json(
                ['message' => 'Error al actualizar la carrera'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        // Procesar la carrera para la respuesta
        $career = $this->processCareerImage($career);

        // Mensaje de actualización realizada
        return response()->json(
            [
                'message' => 'Carrera actualizada correctamente',
                'data' => $career
            ],
            Response::HTTP_OK
        );
    }

    // Mostrar una carrera específica
    public function showCareer($id_career)
    {
        // Buscar la carrera por su ID
        $career = Career::find($id_career);

        if (!$career) {
            // Si no existe, retorna un mensaje de error
            return response()->json(
                ['message' => 'No se encontró la carrera'],
                Response::HTTP_NOT_FOUND
            );
        }

        // Procesar la carrera para la respuesta
        $career = $this->processCareerImage($career);

        // Devolver la respuesta con la carrera
        return response()->json([
            'message' => 'Carrera encontrada',
            'data' => $career,
        ], Response::HTTP_OK);
    }

    // Método auxiliar para procesar la imagen de la carrera
    private function processCareerImage($career)
    {
        // Agregar la URL completa de la imagen
        $career->image_url = $career->image_career ? url(Storage::url($career->image_career)) : null;
        
        // Si no hay Base64 guardado pero hay imagen en storage, generarlo
        if (empty($career->base64_image) && $career->image_career && Storage::disk('public')->exists($career->image_career)) {
            $imageData = Storage::disk('public')->get($career->image_career);
            $career->base64_image = 'data:image/' . pathinfo($career->image_career, PATHINFO_EXTENSION) . ';base64,' . base64_encode($imageData);
        }

        return $career;
    }
}