<?php

namespace App\Http\Controllers\Api;

use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use App\Models\Admin;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class AdminController extends Controller
{
   public function listAdmin(Request $request)
{

    $perPage = $request->input('per_page', 20);


    $admins = Admin::paginate($perPage);

    if ($admins->isEmpty()) {

        return response()->json(
            ['message' => 'No se encontró el admin'],
            Response::HTTP_NOT_FOUND
        );
    }


    return response()->json(
        $admins,
        Response::HTTP_OK
    );
}

    public function RegisterAdmins(Request $request)
    {
        //Valida los datos a registrar
        $validator = Validator::make(
            $request->all(),
            [
                'nombre_admins' => 'required|max:100',
                'apellido_admins' => 'required|max:100',
                'ci_admins' => 'required|max:30|unique:admins_life',
                'correo_admins' => 'required|max:100|email|unique:admins_life',
                'password_admins' => 'required|max:61',
                'status_admins' => 'required|max:50',
                'id_paises' => 'required',
                'id_permissions' => 'required'
            ]
        );

        //Si el registro falla
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al registrar el admin',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        //Si la validacion es correcta creo

        $admin = Admin::create([
            'nombre_admins' => $request->nombre_admins,
            'apellido_admins' => $request->apellido_admins,
            'ci_admins' => $request->ci_admins,
            'correo_admins' => $request->correo_admins,
            'password_admins' => Hash::make($request->password_admins),
            'status_admins' => $request->status_admins,
            'id_paises' => $request->id_paises,
            'id_permissions' => $request->id_permissions
        ]);


        //Si el registro no es exitoso
        if (!$admin) {
            return response()->json(
                ['message' => 'Error al guardar los datos'],
                Response::HTTP_INTERNAL_SERVER_ERROR
            );
        }

        //Si el registro es exitoso
        // $token = $admin->createToken('register_admin_token')->plainTextToken;
        return response()->json(
            [
                'message' => 'El usuario fue registrado correctamente',
                'data' => $admin,
                // 'access_token' => $token,
                // 'token_type' => 'Bearer'
            ],
            Response::HTTP_CREATED
        );
    }

    public function ShowAdmins($id_admins)
    {
        $admin = Admin::find($id_admins);
        if (!$admin) {
            //si no existe retorna un mensaje de error
            return response()->json(
                ['message' => 'No se encontro el admin'],
                Response::HTTP_NOT_FOUND
            );
        }

        //si existe retorna un mensaje con los datos solicitados
        return response()->json(
            ['message' => $admin],
            Response::HTTP_OK
        );
    }

    public function UpgradePartialAdmins(Request $request, $id_admins)
    {
        //busqueda por id
        $admin = Admin::find($id_admins);
        if (!$admin) {
            //si no existe retorna un mensaje de error
            return response()->json(
                ['message' => 'No se encontro el usuario'],
                Response::HTTP_NOT_FOUND
            );
        }

        //valido los datos a actualizar
        $validator = Validator::make($request->all(), [
            'nombre_admins' => 'required|max:100',
            'apellido_admins' => 'required|max:100',
            'ci_admins' => 'required|max:30',
            'correo_admins' => 'required|max:100|email|unique:admins_life',
            'password_admins' => 'required|max:61',
            'status_admins' => 'required|max:50',
            'id_paises' => 'required',
            'id_permissions' => 'required'
        ]);

        //si los datos son invalidos mensaje de error
        if ($validator->fails()) {
            return response()->json(
                [
                    'message' => 'Error al actualizar el admin',
                    'errors' => $validator->errors(),
                ],
                Response::HTTP_BAD_REQUEST
            );
        }

        // solicitud de los datos a actualizar
        if ($request->has('nombre_admins')) {
            $admin->nombre_admins = $request->nombre_admins;
        }

        if ($request->has('apellido_admins')) {
            $admin->apellido_admins = $request->apellido_admins;
        }

        if ($request->has('ci_admins')) {
            $admin->ci_user = $request->ci_user;
        }

        if ($request->has('correo_admins')) {
            $admin->correo_admins = $request->correo_admins;
        }

        if ($request->has('password_admins')) {
            $admin->password_admins = $request->password_admins;
        }

        if ($request->has('status_admins')) {
            $admin->status_admins = $request->status_admins;
        }

        if ($request->has('id_paises')) {
            $admin->id_paises = $request->id_paises;
        }

        if ($request->has('id_permissions')) {
            $admin->id_permissions = $request->id_permissions;
        }


        //guardar los datos suministrados
        $admin->save();

        //mensaje de actualizacion realizada
        return response()->json(
            ['message' => $admin],
            Response::HTTP_OK
        );
    }

    public function validateLoginAdmin(Request $request)
    {
        // Validar la entrada
        $validator = Validator::make($request->all(), [
            'correo_admins' => 'required|email',
            'password_admins' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validación fallida',
                'errors' => $validator->errors()
            ], Response::HTTP_UNPROCESSABLE_ENTITY); // 422 Unprocessable Entity
        }

        // Buscar el usuario por correo
        $admin = Admin::where('correo_admins', $request->correo_admins)->first();


        // Verificar si el usuario existe y si la contraseña es correcta
        if (!$admin || !Hash::check($request->password_admins, $admin->password_admins)) {
            return response()->json([
                'success' => false,
                'message' => 'Credenciales incorrectas'
            ], Response::HTTP_UNAUTHORIZED); // 401 Unauthorized
        }


        $token = $admin->createToken('login_admin_token' . $admin->nombre_admins, [], now()->addMinutes(config('sanctum.expiration')))->plainTextToken;



        // Retornar la respuesta
        return response()->json([
            'success' => true,
            'message' => 'Bienvenido ' . $admin->nombre_admins,
            'data' => $admin,
            'access_token' => $token,
            'token_type' => 'Bearer',
            'expires_in' => config('sanctum.expiration') * 60, // Convertir a segundos
        ], Response::HTTP_OK); // 200 OK
    }

    public function logoutAdmin(Request $request)
    {
        // Verifica si el usuario está autenticado
        if (!Auth::check()) {
            return response()->json(['message' => 'No estás autenticado.'], Response::HTTP_UNAUTHORIZED);
        }

        // Obtiene el token actual del usuario
        $token = $request->user()->currentAccessToken();

        // Verifica si el token existe
        if ($token) {
            // Elimina el token
            $token->delete();
            return response()->json(['message' => 'Logout exitoso.'], Response::HTTP_OK);
        }

        return response()->json(['message' => 'No se pudo cerrar la sesión.'], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
